import { Suspense } from "react";
import Link from "next/link";
import { notFound } from "next/navigation";
import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import InnerBanner from "@/components/InnerBanner";
import "../blog.css";
import { Metadata } from "next";

export const dynamic = 'force-dynamic';

export interface BlogPost {
    id: string | number;
    slug: string;
    title: string;
    excerpt: string;
    short: string;
    content: string;
    image: string;
    coverImage: string;
    category: string;
    author: string;
    date: string;
    readTime: string;
    meta_title?: string;
    meta_description?: string;
    meta_keywords?: string;
}

interface BlogPostContentProps {
    post: BlogPost;
    relatedBlogs?: BlogPost[];
}

function PageLoader() {
    return (
        <div className="min-vh-100 d-flex align-items-center justify-content-center">
            <div className="spinner-border text-primary" />
        </div>
    );
}

import { buildBlogPostMeta } from "@/lib/seo/blogPostMeta";

async function fetchBlogPost(slug: string): Promise<BlogPost> {
    const res = await API_ENDPOINTS.BLOGS.DETAIL(slug);
    if (!res.ok) throw new Error(`Failed to fetch blog: ${res.status}`);
    return res.json();
}

async function fetchBlogs(): Promise<BlogPost[]> {
    try {
        const res = await API_ENDPOINTS.BLOGS.LIST();
        if (!res.ok) return [];
        return res.json();
    } catch {
        return [];
    }
}

export async function generateMetadata(
    { params }: { params: Promise<{ slug: string }> }
): Promise<Metadata> {

    const { slug } = await params;
    const normalizedSlug = decodeURIComponent(slug).trim().toLowerCase();

    try {
        const res = await API_ENDPOINTS.BLOGS.DETAIL(normalizedSlug);
        if (!res.ok) throw new Error("Blog not found");

        const post = await res.json();

        return buildBlogPostMeta(post, normalizedSlug);

    } catch {
        return {
            title: "Blog Post | Excelsius IGNOU",
            description: "Read our latest blog articles and guides.",
        };
    }
}

// --------------------- CONTENT COMPONENT ---------------------
function BlogPostContent({ post, relatedBlogs = [] }: BlogPostContentProps) {
    if (!post) {
        return (
            <div className="container py-5 text-center">
                <h2>Blog post not found</h2>
            </div>
        );
    }

    const formatDate = (dateString: string) => {
        try {
            const date = new Date(dateString);
            return date.toLocaleDateString('en-US', {
                year: 'numeric',
                month: 'long',
                day: 'numeric'
            });
        } catch {
            return dateString;
        }
    };

    return (
        <>
            <InnerBanner
                title={post.title}
                breadcrumb={`Blog / ${post.title}`}
            />

            <div className="blog-content-section pt-5">
                <div className="container">
                    <div className="row">
                        {/* MAIN CONTENT */}
                        <div className="col-lg-8">
                            <img
                                src={post.image}
                                alt={post.title}
                                className="main-image"
                                style={{
                                    width: '100%',
                                    height: '500px',
                                    objectFit: 'cover',
                                    borderRadius: '12px'
                                }}
                            />
                            <hr className="divider" />
                            <h1 className="blog-title" style={{
                                fontSize: '42px',
                                fontWeight: '700',
                                color: '#222',
                                marginBottom: '20px'
                            }}>
                                {post.title}
                            </h1>

                            <div style={{
                                fontSize: '16px',
                                lineHeight: '1.8',
                                color: '#555',
                                marginBottom: '20px'
                            }}>
                                {post.content ? post.content
                                    .split('\n\n')
                                    .map((paragraph, index) => (
                                        <p key={index} style={{ marginBottom: '20px' }} dangerouslySetInnerHTML={{ __html: paragraph }} />
                                    )) : null}
                            </div>

                            <hr className="divider" />
                            <Link
                                href="/blog"
                                className="back-btn"
                                style={{
                                    display: 'inline-flex',
                                    alignItems: 'center',
                                    gap: '8px',
                                    color: '#000',
                                    textDecoration: 'none',
                                    marginBottom: '30px'
                                }}
                            >
                                ← Back to All Articles
                            </Link>
                        </div>

                        {/* RELATED BLOGS SIDEBAR */}
                        <div className="col-lg-4">
                            <div className="sidebar-section">
                                <h3 className="sidebar-title" style={{
                                    fontSize: '24px',
                                    fontWeight: '700',
                                    marginBottom: '25px'
                                }}>
                                    Related Articles
                                </h3>

                                {relatedBlogs.length > 0 ? (
                                    relatedBlogs.map((blog) => (
                                        <Link
                                            key={blog.id}
                                            href={`/blog/${blog.slug}`}
                                            className="related-blog-link"
                                            style={{ textDecoration: 'none' }}
                                        >
                                            <div className="related-blog-card" style={{
                                                background: '#fff',
                                                borderRadius: '12px',
                                                overflow: 'hidden',
                                                marginBottom: '20px',
                                                border: '1px solid #ddd',
                                                display: 'flex'
                                            }}>
                                                <img
                                                    src={blog.image}
                                                    alt={blog.title}
                                                    className="related-blog-img"
                                                    style={{
                                                        width: '120px',
                                                        height: '120px',
                                                        objectFit: 'cover'
                                                    }}
                                                />
                                                <div className="related-blog-content" style={{
                                                    padding: '15px',
                                                    flex: 1
                                                }}>
                                                    <h4 className="related-blog-title" style={{
                                                        fontSize: '16px',
                                                        fontWeight: '600',
                                                        marginBottom: '10px',
                                                        color: '#222'
                                                    }}>
                                                        {blog.title}
                                                    </h4>
                                                    <div className="related-blog-date" style={{
                                                        fontSize: '14px',
                                                        color: '#999'
                                                    }}>
                                                        {formatDate(blog.date)}
                                                    </div>
                                                </div>
                                            </div>
                                        </Link>
                                    ))
                                ) : (
                                    <p style={{ color: '#999' }}>No Related Articles</p>
                                )}
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </>
    );
}

export default async function BlogPostPage({
    params
}: {
    params: Promise<{ slug: string }>;
}) {
    const { slug } = await params;
    const normalizedSlug = decodeURIComponent(slug).trim().toLowerCase();

    try {
        const post = await fetchBlogPost(normalizedSlug);
        const allBlogs = await fetchBlogs();

        const relatedBlogs = allBlogs
            .filter(blog => blog.id !== post.id && blog.slug !== normalizedSlug)
            .filter(blog => blog.title && blog.image)
            .slice(0, 3); // Max 3

        return (
            <Suspense fallback={<PageLoader />}>
                <BlogPostContent post={post} relatedBlogs={relatedBlogs} />
            </Suspense>
        );
    } catch (error) {
        notFound();
    }
}
